--========================================================================================================
-- qb-animalzones (SERVER) Written per ILoveNopixelILoveYou (Highleak.com)
--========================================================================================================
local OnesyncEnabled = (GetConvarInt('onesync_enabled', 0) == 1 or GetConvarInt('onesync', 0) == 1)
if not OnesyncEnabled then
    print("^3[qb-animalzones]^7 ATTENTION: OneSync n'est pas activé. Le script est prévu pour OneSync (compat).")
end

--===========================================================
-- État des zones
--===========================================================
local Zones = {}

-- Merge extra exempt jobs from Config.MedicalBridge without changing the base list
if Config and Config.MedicalBridge and Config.MedicalBridge.ExtraExemptJobs then
    -- Robustness: ensure base table exists even if not defined in config
    Config.ExemptJobs = Config.ExemptJobs or {}
    local set = {}
    for _, j in ipairs(Config.ExemptJobs or {}) do set[j] = true end
    for _, j in ipairs(Config.MedicalBridge.ExtraExemptJobs) do
        if not set[j] then table.insert(Config.ExemptJobs, j) end
    end
end

for _, z in ipairs(Config.Zones) do
    Zones[z.id] = {
        cfg = z,
        controller = nil,      -- src joueur
        state = 'idle',
        players = {},          -- [src] = {pos=vector3, job, coma, ts}
    }
end

local function isExemptJob(jobName)
    if not jobName then return false end
    for _, j in ipairs(Config.ExemptJobs or {}) do
        if j == jobName then return true end
    end
    return false
end

--===========================================================
-- Heartbeat des clients
--===========================================================
RegisterNetEvent('qb-animalzones:server:heartbeat', function(payload)
    local src = source
    if type(payload) ~= 'table' then return end

    -- OneSync sanity: if onesync is disabled or player not active, fail fast
    local onesync = GetConvar('onesync', 'off')
    if onesync == 'off' then
        -- We still allow logic to run in single-player environments, but avoid any network assumptions
    end

    -- Always trust server-side position, not client payload
    local ped = GetPlayerPed(src)
    if not ped or ped == 0 then return end
    local pos = GetEntityCoords(ped)

    -- Copy selected fields from payload but sanitize types
    local job = (type(payload.job) == 'string' and payload.job) or 'unemployed'
    local coma = payload.coma and true or false
    local inside = (type(payload.inside) == 'table' and payload.inside) or {}
    local near = (type(payload.near) == 'table' and payload.near) or {}
    local ts = (type(payload.ts) == 'number' and payload.ts) or 0

    -- Your original heartbeat logic should follow here, but use `pos` computed server-side.
    if not pos then return end

    -- Example: stash last-known position server-side for zone checks
    if not _G.QB_ANIMALZONES then _G.QB_ANIMALZONES = {} end
    _G.QB_ANIMALZONES[src] = _G.QB_ANIMALZONES[src] or {}
    _G.QB_ANIMALZONES[src].pos = pos
    _G.QB_ANIMALZONES[src].job = job
    _G.QB_ANIMALZONES[src].coma = coma
    _G.QB_ANIMALZONES[src].inside = inside
    _G.QB_ANIMALZONES[src].near = near
    _G.QB_ANIMALZONES[src].ts = ts

    
    -- Update per-zone presence using authoritative server position
    for zoneId, Z in pairs(Zones) do
        local cfg = Z.cfg
        local dist = #(pos - cfg.center)
        local farMargin = (Config.DespawnFarMargin or 250.0)
        if dist <= (cfg.radius + farMargin) then
            Z.players[src] = { pos = pos, job = job, coma = coma, ts = GetGameTimer() }
        else
            Z.players[src] = nil
        end
    end

    -- (rest of your server logic continues, ensuring server-side coords are authoritative)
end)

AddEventHandler('playerDropped', function()
    local src = source
    for zoneId, Z in pairs(Zones) do
        Z.players[src] = nil
        if Z.controller == src then
            -- Le contrôleur vient de quitter : laisser OneSync migrer les entités et
            -- permettre au prochain contrôleur d'adopter les animaux existants (pas de pop/despawn brutal).
            Z.controller = nil
            -- L'état sera recalculé au prochain tick
        end
    end
end)

--===========================================================
-- Tick serveur principal
--===========================================================
CreateThread(function()
    while true do
        local now = GetGameTimer()
        for zoneId, Z in pairs(Zones) do
            local cfg = Z.cfg
            local nearList, insideList, farList, nonComaInside = {}, {}, {}, {}
            local stale = (Config.StaleHeartbeatMs or 5000)

            -- Collecter présences fraîches
            for src, info in pairs(Z.players) do
                if (now - (info.ts or 0)) <= stale then
                    local dist = #(info.pos - cfg.center)
                    if dist <= cfg.radius then
                        table.insert(insideList, {src=src, info=info, dist=dist})
                        if not info.coma then
                            table.insert(nonComaInside, {src=src, info=info, dist=dist})
                        end
                    elseif dist <= (cfg.radius + (Config.PreSpawnBuffer or 120.0)) then
                        table.insert(nearList, {src=src, info=info, dist=dist})
                    elseif dist <= (cfg.radius + (Config.DespawnFarMargin or 250.0)) then
                        table.insert(farList, {src=src, info=info, dist=dist})
                    end
                end
            end

            table.sort(insideList, function(a,b) return a.dist < b.dist end)
            table.sort(nearList,  function(a,b) return a.dist < b.dist end)
            table.sort(farList,   function(a,b) return a.dist < b.dist end)

            local hasExempt = false
            for _, e in ipairs(insideList) do
                if isExemptJob(e.info.job) then hasExempt = true break end
            end

            -- Déterminer l'état désiré
            local desired = 'idle'
            if hasExempt then
                desired = 'blocked' -- [#13]
            else
                if #insideList > 0 then
                    if #nonComaInside == 0 then
                        desired = 'despawn' -- tous en coma -> despawn [#12]
                    else
                        desired = 'active' -- attaque [#3,#6]
                    end
                elseif #nearList > 0 then
                    desired = 'pre' -- joueurs proches mais pas dedans -> pré-spawn [#5]
                elseif #farList > 0 then
                    desired = 'idle' -- contrôle conservé: attente invincible [#7]
                else
                    desired = 'idle' -- sans contrôleur (sera libéré plus bas) [#7]
                end
            end

            -- Choix du contrôleur
            local newController = Z.controller
            if desired == 'active' or desired == 'pre' or desired == 'idle' then
                -- sélectionner le plus proche du centre selon la meilleure liste dispo
                local cand = (#insideList > 0) and insideList or ((#nearList > 0) and nearList or farList)
                if cand[1] then
                    newController = cand[1].src
                    -- borne de sécurité : ne pas donner le contrôle si trop loin [Config.ControllerMaxDistance]
                    local maxCtrl = (cfg.radius + (Config.ControllerMaxDistance or 800.0))
                    if cand[1].dist > maxCtrl then
                        newController = nil
                    end
                else
                    newController = nil
                end
            else
                newController = nil
            end

            -- Transitions de contrôleur
            if newController ~= Z.controller then
                if Z.controller then
                    -- libérer l'ancien (s'il est encore connecté)
                    TriggerClientEvent('qb-animalzones:client:releaseControl', Z.controller, zoneId)
                end
                Z.controller = nil
                if newController then
                    TriggerClientEvent('qb-animalzones:client:takeControl', newController, zoneId, cfg)
                    Z.controller = newController
                end
            end

            -- Propager l'état à l'actuel contrôleur
            if Z.controller then
                if Z.state ~= desired then
                    Z.state = desired
                    TriggerClientEvent('qb-animalzones:client:setState', Z.controller, zoneId, desired)
                end
            else
                Z.state = 'idle'
            end

            -- sécurité: si plus personne dans le voisinage élargi -> libérer et despawn [#7]
            if (not nearList[1]) and (not insideList[1]) and (not farList[1]) then
                if Z.controller then
                    TriggerClientEvent('qb-animalzones:client:releaseControl', Z.controller, zoneId)
                end
                Z.controller = nil
                Z.state = 'idle'
            end
        end

        Wait(Config.TickIntervalMs or 1000)
    end
end)


-- Safe delete utility to handle possible network hiccups
function AZ_TryDeleteEntity(ent)
    if not ent or ent == 0 then return end
    if NetworkGetEntityIsNetworked(ent) then
        local owner = NetworkGetEntityOwner(ent)
        if owner and owner ~= -1 then
            -- request control (server-side hint)
            NetworkRequestControlOfEntity(ent)
        end
    end
    DeleteEntity(ent)
end
