--========================================================================================================
-- qb-animalzones (CLIENT) Written per ILoveNopixelILoveYou (Highleak.com)
--========================================================================================================
-- Safe SetTimeout alias (client) to avoid nil on some builds

-- Safe QBCore getter (works even if not globally available)
local QBCore = rawget(_G, 'QBCore')
if not QBCore then
    local ok, obj = pcall(function()
        return exports['qb-core'] and exports['qb-core']:GetCoreObject()
    end)
    if ok and obj then QBCore = obj end
end

local SetTimeout = SetTimeout or (Citizen and Citizen.SetTimeout) or function(ms, cb)
    CreateThread(function() Wait(ms) if type(cb) == 'function' then pcall(cb) end end)
end


local DEBUG = Config.Debug or false
local function debug(msg) if DEBUG then print(('[qb-animalzones][client] %s'):format(msg)) end end

---@type table<string, {cfg:table, animals:table, state:string, controller:boolean, lastAttackRefresh:number}>
local ClientZones = {}

--===========================================================
-- Hostilité globale animaux <-> joueurs (agression) [#3]
--===========================================================
local function initRelationships()
    AddRelationshipGroup('WILD_ANIMALS')
    -- joueurs
    SetRelationshipBetweenGroups(5, `PLAYER`, GetHashKey('WILD_ANIMALS')) -- Hate
    SetRelationshipBetweenGroups(5, GetHashKey('WILD_ANIMALS'), `PLAYER`)
    -- ped AI tuning pour le joueur local
    local p = PlayerPedId()
    if DoesEntityExist(p) then
        SetPedRelationshipGroupHash(p, `PLAYER`)
    end
end

CreateThread(function()
    initRelationships()
end)

-- Réappliquer lors du (re)spawn du joueur
AddEventHandler('playerSpawned', function() initRelationships() end)
RegisterNetEvent('QBCore:Client:OnPlayerLoaded', function() initRelationships() end)

--===========================================================
-- Aides QBCore / état joueur [#11,#12]
--===========================================================
local function getPlayerData()
    if QBCore and QBCore.Functions and QBCore.Functions.GetPlayerData then
        return QBCore.Functions.GetPlayerData()
    end
    return {}
end

local function getJobName()
    local data = getPlayerData()
    if data and data.job and data.job.name then
        return data.job.name
    end
    return "unemployed"
end

local function isPlayerComa()
    -- Bridge call (supports alternative ambulance jobs). If it returns a boolean, trust it.
    if type(AnimalZones_IsPlayerComaBridge) == 'function' then
        local ok, bridged = pcall(AnimalZones_IsPlayerComaBridge)
        if ok and bridged ~= nil then
            return bridged and true or false
        end
    end
    -- 1) Essayer via metadata QBCore si dispo
    local data = getPlayerData()
    if data and data.metadata then
        for _, key in ipairs(Config.ComaMetaKeys or {}) do
            local v = data.metadata[key]
            if v == true then return true end
        end
    end
    -- 2) qb-hospital exports (si présent)
    local okDead, isDead = pcall(function()
        return exports["qb-hospital"] and exports["qb-hospital"]:GetDeathStatus()
    end)
    if okDead and isDead then return true end
    local okLast, inLS = pcall(function()
        return exports["qb-hospital"] and exports["qb-hospital"]:IsInLaststand()
    end)
    if okLast and inLS then return true end
    -- 3) Natives GTA en dernier recours
    local ped = PlayerPedId()
    if IsPedFatallyInjured(ped) or IsPedDeadOrDying(ped) then
        return true
    end
    return false
end

-- Détection coma pour AUTRES joueurs via statebags (si dispo)
-- Robust coma detection for OTHER players (never uses IsPedInjured)
local function isOtherPedComa(ped)
    if not ped or not DoesEntityExist(ped) then return false end

    -- Prefer EMS bridge if available
    if type(IsPlayerComa) == 'function' then
        -- we only have a PED, not server id; fallback continues
    end

    -- Try statebags on the entity
    local ok, state = pcall(function() return Entity(ped).state end)
    if ok and state then
        local keys = {
            'isDead','dead','isdead',
            'inLaststand','isInLaststand','inlaststand'
        }
        if Config and Config.MedicalBridge and Config.MedicalBridge.CustomStateKeys then
            for _,k in ipairs(Config.MedicalBridge.CustomStateKeys) do keys[#keys+1]=k end
        end
        for _,k in ipairs(keys) do
            local v = state[k]
            if v ~= nil then
                if type(v) == 'boolean' then return v end
                if type(v) == 'number' then return v ~= 0 end
                if type(v) == 'string' then
                    local s = v:lower()
                    if s == 'true' or s == '1' then return true end
                    if s == 'false' or s == '0' then return false end
                end
            end
        end
    end

    -- Last resort: natives that truly mean "down"
    if IsPedDeadOrDying(ped) or IsPedFatallyInjured(ped) then return true end
    return false
end

--===========================================================
-- Aides joueurs [#6]
--===========================================================
local function playersInZone(cfg)
    local res = {}
    local center, within = cfg.center, cfg.radius
    for _, pid in ipairs(GetActivePlayers()) do
        local ped = GetPlayerPed(pid)
        if DoesEntityExist(ped) then
            local coords = GetEntityCoords(ped)
            local dist = #(coords - center)
            if dist <= within then
                res[#res+1] = {pid=pid, ped=ped, coords=coords, dist=dist}
            end
        end
    end
    table.sort(res, function(a,b) return a.dist < b.dist end)
    return res
end

--===========================================================
-- Sol / Z
--===========================================================
local function groundZAt(x, y, zHint)
    if not Config.SpawnZProbe then
        return zHint or 0.0
    end
    local success, z = GetGroundZFor_3dCoord(x+0.0, y+0.0, (zHint or 1000.0)+0.0, false)
    if success then return z end
    return zHint or 0.0
end

--===========================================================
-- Détection & adoption des animaux "orphelins" (changement de contrôleur)
--===========================================================
local function adoptExistingAnimals(Z)
    local cfg = Z.cfg
    if not cfg or not cfg.animalModel then return end
    local adopted = 0

    local handle, ped = FindFirstPed()
    local success
    repeat
        if DoesEntityExist(ped) and not IsPedAPlayer(ped) then
            -- Même modèle et dans le périmètre de la zone
            if GetEntityModel(ped) == cfg.animalModel then
                local pos = GetEntityCoords(ped)
                if #(pos - cfg.center) <= (cfg.spawnRadius + 4.0) then
                    -- Vérifier que ce ped n'est pas déjà dans notre table
                    local known = false
                    for _, a in ipairs(Z.animals) do
                        if a.ped == ped then known = true break end
                    end
                    if not known then
                        -- S'assurer du bon groupe & flags
                        SetPedRelationshipGroupHash(ped, GetHashKey('WILD_ANIMALS'))
                        SetEntityAsMissionEntity(ped, true, true)
                        table.insert(Z.animals, {ped=ped, spawnTs=GetGameTimer(), deathTs=nil})
                        adopted = adopted + 1
                    end
                end
            end
        end
        success, ped = FindNextPed(handle)
    until not success
    EndFindPed(handle)
    if adopted > 0 then
        debug(("zone %s: adoption de %d animaux existants"):format(cfg.id or "?", adopted))
    end
end

--===========================================================
-- Apparitions & points [#2]
--===========================================================
local function sampleSpawnPoints(cfg, count, Z)
    local points = {}
    local tries, maxTries = 0, count * 40
    while #points < count and tries < maxTries do
        tries = tries + 1
        local a = math.random() * math.pi * 2.0
        local r = (cfg.spawnRadius - 8.0) * math.sqrt(math.random())
        local x = cfg.center.x + math.cos(a) * r
        local y = cfg.center.y + math.sin(a) * r
        local z = groundZAt(x, y, cfg.center.z)
        local candidate = vector3(x,y,z)

        -- Anti pop-in : pas trop proche d'un joueur
        local tooClosePlayer = false
        for _, pid in ipairs(GetActivePlayers()) do
            local pp = GetPlayerPed(pid)
            if DoesEntityExist(pp) then
                local ppos = GetEntityCoords(pp)
                if #(ppos - candidate) < (Config.MinSpawnToPlayer or 65.0) then
                    tooClosePlayer = true
                    break
                end
            end
        end
        if tooClosePlayer then goto continue end

        -- Séparation entre nouveaux points
        local sepOK = true
        for _, p in ipairs(points) do
            if #(p - candidate) < (Config.MinSpawnSeparation or 18.0) then
                sepOK = false
                break
            end
        end
        if not sepOK then goto continue end

        -- Séparation aussi par rapport aux animaux déjà présents
        if Z and Z.animals then
            for _, a in ipairs(Z.animals) do
                if a.ped and DoesEntityExist(a.ped) then
                    local ap = GetEntityCoords(a.ped)
                    if #(ap - candidate) < (Config.MinSpawnSeparation or 18.0) then
                        sepOK = false
                        break
                    end
                end
            end
            if not sepOK then goto continue end
        end

        points[#points+1] = candidate
        ::continue::
    end
    return points
end

--===========================================================
-- Création / suppression
--===========================================================
local function configureAnimalPed(ped)
    SetBlockingOfNonTemporaryEvents(ped, true)
    SetPedCanRagdoll(ped, true)
    SetPedCanRagdollFromPlayerImpact(ped, true)
    SetEntityAsMissionEntity(ped, true, true)
    SetPedRelationshipGroupHash(ped, GetHashKey('WILD_ANIMALS'))
    SetPedHearingRange(ped, Config.AnimalMaxHearing or 80.0)
    SetPedSeeingRange(ped, Config.AnimalMaxSight or 85.0)
    SetPedFleeAttributes(ped, 0, false) -- ne fuit pas
    SetPedCombatAttributes(ped, 46, true) -- AlwaysFight
    SetPedCombatAttributes(ped, 5, true)  -- CanFightArmedPedsWhenNotArmed
    SetPedCombatAbility(ped, 2)           -- high
    SetPedCombatRange(ped, 2)             -- far
    SetPedAlertness(ped, 3)
    if Config.AntiSlideOnSpawn then
        SetEntityVelocity(ped, 0.0, 0.0, 0.0)
    end
end

local function setInvincible(ped, inv)
    SetEntityInvincible(ped, inv)
    SetEntityCanBeDamaged(ped, not inv)
end

-- Santé personnalisée selon la config [#14]
local function setHealthFromConfig(ped, model)
    local hp = nil
    if Config.HealthPerModel then
        hp = Config.HealthPerModel[model]
        if not hp then
            local m = GetEntityModel(ped)
            hp = Config.HealthPerModel[m]
        end
    end
    if not hp then hp = Config.DefaultAnimalHealth end
    if hp and hp > 0 then
        if SetEntityMaxHealth then SetEntityMaxHealth(ped, hp) end
        SetEntityHealth(ped, hp)
    end
end

local function createAnimal(model, at)
    if not IsModelInCdimage(model) or not IsModelValid(model) then
        debug(("model %s invalid"):format(tostring(model)))
        return nil
    end
    RequestModel(model)
    local t = GetGameTimer() + 5000
    while not HasModelLoaded(model) do
        Wait(0)
        if GetGameTimer() > t then
            debug(("model %s failed to load"):format(model))
            return nil
        end
    end
    local ped = CreatePed(28, model, at.x, at.y, at.z, math.random()*360.0, true, true) -- 28 = animal
    setHealthFromConfig(ped, model)
    if Config.AntiSlideOnSpawn then SetEntityVelocity(ped, 0.0, 0.0, 0.0) end
    -- OneSync: enregistrer en réseau [#11]
    if not NetworkGetEntityIsNetworked(ped) then
        NetworkRegisterEntityAsNetworked(ped)
    end
    local netId = NetworkGetNetworkIdFromEntity(ped)
    SetNetworkIdCanMigrate(netId, true)
    SetNetworkIdExistsOnAllMachines(netId, true)

    configureAnimalPed(ped)
    -- comportement de pré-attente
    -- ensure clear tasks on fresh spawn
    ClearPedTasksImmediately(ped)
    TaskStandStill(ped, 1500)
    setInvincible(ped, true) -- par défaut invincible (pré-spawn)

    SetModelAsNoLongerNeeded(model)
    return ped
end

local function safeDelete(ped)
    if not ped or not DoesEntityExist(ped) then return end
    -- Essayer de prendre le contrôle réseau avant suppression
    local timeout = GetGameTimer() + 1000
    while not NetworkHasControlOfEntity(ped) and GetGameTimer() < timeout do
        NetworkRequestControlOfEntity(ped)
        Wait(0)
    end
    SetEntityAsMissionEntity(ped, true, true)
    DeleteEntity(ped)
end

local function deleteAnimal(ped)
    safeDelete(ped)
end

--===========================================================
-- Contrôle zone (client)
--===========================================================
local function clampZoneCfg(cfg)
    -- sécurité : spawnRadius ne doit pas dépasser radius
    if cfg and cfg.radius and cfg.spawnRadius then
        if cfg.spawnRadius > cfg.radius then
            cfg.spawnRadius = math.max(0.0, cfg.radius - 2.0)
        end
    end
end

local function ensureZoneInstance(zoneId, cfg)
    if not ClientZones[zoneId] then
        ClientZones[zoneId] = {
            cfg = cfg or {},
            animals = {},
            state = 'idle',
            controller = false,
            lastAttackRefresh = 0
        }
    else
        if cfg and next(cfg) ~= nil then
            ClientZones[zoneId].cfg = cfg
        end
    end
    clampZoneCfg(ClientZones[zoneId].cfg)
    return ClientZones[zoneId]
end

local function setAnimalsInvincible(Z, inv)
    for _, a in ipairs(Z.animals) do
        local ped = a.ped
        if ped and DoesEntityExist(ped) then
            setInvincible(ped, inv)
        end
    end
end

local function clearCombatTasks(Z)
    -- Garantit l'arrêt immédiat des attaques quand on repasse en pré/idle [#7]
    for _, a in ipairs(Z.animals) do
        local ped = a.ped
        if ped and DoesEntityExist(ped) and not IsEntityDead(ped) then
            ClearPedTasks(ped)
        end
    end
end

local function zoneEnsurePopulation(Z)
    local cfg = Z.cfg
    -- adopter ce qui existe déjà pour éviter les doublons lors d'un changement de contrôleur
    -- Limiter les scans d'adoption pour des raisons de perfs
    local now = GetGameTimer()
    if not Z.adoptLastScan or (now - Z.adoptLastScan) >= (Config.AdoptScanMs or 6000) then
        adoptExistingAnimals(Z)
        Z.adoptLastScan = now
    end

    local want = math.max(0, cfg.animalCount or 0)
    local have = 0
    for _, a in ipairs(Z.animals) do
        if a.ped and DoesEntityExist(a.ped) and not IsEntityDead(a.ped) then
            have = have + 1
        end
    end
    local toMake = math.min(want - have, Config.SpawnPerTick or 2)
    if toMake <= 0 then return end

    local pts = sampleSpawnPoints(cfg, toMake, Z)
    for i=1, #pts do
        local ped = createAnimal(cfg.animalModel, pts[i])
        if ped then
            local entry = {ped=ped, spawnTs=GetGameTimer(), deathTs=nil}
            table.insert(Z.animals, entry)
        end
    end
end

-- CORRECTION: ne jamais éjecter les cadavres de la table afin de respecter la persistance 2 min [#4]
local function zoneCull(Z)
    local cfg = Z.cfg
    local want = math.max(0, cfg.animalCount or 0)

    -- Compter les vivants
    local living = 0
    for _, a in ipairs(Z.animals) do
        if a.ped and DoesEntityExist(a.ped) and not IsEntityDead(a.ped) then
            living = living + 1
        end
    end
    local excess = living - want
    if excess <= 0 then return end

    -- Supprimer les plus anciens vivants en priorité, garder les cadavres suivis pour cleanup
    local i = 1
    while excess > 0 and i <= #Z.animals do
        local a = Z.animals[i]
        if a and a.ped and DoesEntityExist(a.ped) and not IsEntityDead(a.ped) then
            deleteAnimal(a.ped)
            table.remove(Z.animals, i)
            excess = excess - 1
            -- ne pas incrémenter i, on vient de décaler le tableau
        else
            i = i + 1
        end
    end
end

local function zoneCleanupCadavers(Z)
    local keepMs = (Config.CadaverDespawnMs or 120000)
    local now = GetGameTimer()
    for i = #Z.animals, 1, -1 do
        local a = Z.animals[i]
        if a.ped and DoesEntityExist(a.ped) then
            if IsEntityDead(a.ped) then
                a.deathTs = a.deathTs or now
                if (now - a.deathTs) >= keepMs then
                    deleteAnimal(a.ped)
                    table.remove(Z.animals, i)
                end
            else
                -- vivant ok
            end
        else
            table.remove(Z.animals, i)
        end
    end
end

local function idleBehavior(Z)
    local cfg = Z.cfg
    local now = GetGameTimer()
    local sep = (Config.MinSpawnSeparation or 18.0)

    -- Petite fonction utilitaire: point aléatoire proche mais dans la zone
    local function randomNearbyPoint(origin, maxStep)
        local tries = 0
        while tries < 8 do
            tries = tries + 1
            local ang = math.random() * math.pi * 2.0
            local dist = math.random(4, maxStep or 10) + 0.0
            local offset = vector3(math.cos(ang) * dist, math.sin(ang) * dist, 0.0)
            local p = origin + offset
            -- rester dans le spawnRadius
            if #(p - cfg.center) <= (cfg.spawnRadius - 4.0) then
                -- caler sur le sol
                p = vector3(p.x, p.y, groundZAt(p.x, p.y, cfg.center.z))
                return p
            end
        end
        return origin
    end

    -- L'idle riche boucle sur chaque animal
    for _, a in ipairs(Z.animals) do
        local ped = a.ped
        if ped and DoesEntityExist(ped) and not IsEntityDead(ped) then
            -- Ne pas perturber un combat
            if IsPedInCombat(ped, 0) then goto continue end

            local pos = GetEntityCoords(ped)

            -- 1) Légères rotations / inspection (regards aléatoires)
            a._nextTurn = a._nextTurn or (now + math.random(1500, 3500))
            if now >= a._nextTurn then
                local h = GetEntityHeading(ped) + math.random(-55, 55) + 0.0
                TaskStandStill(ped, math.random(800, 1600))
                TaskAchieveHeading(ped, h, math.random(900, 1600))
                -- Chance de regarder un point proche
                if math.random() < 0.45 then
                    local look = randomNearbyPoint(pos, 10)
                    TaskLookAtCoord(ped, look.x, look.y, look.z + 0.5, math.random(800, 1600), 0, 2)
                end
                a._nextTurn = now + math.random(1800, 4200)
            end

            -- 2) Petites marches erratiques (espacement passif)
            a._nextStep = a._nextStep or (now + math.random(2200, 5200))
            if now >= a._nextStep then
                -- si trop proche d'un autre animal, s'écarter
                local needSpace = false
                local away = vector3(0.0, 0.0, 0.0)
                for _, b in ipairs(Z.animals) do
                    if b ~= a and b.ped and DoesEntityExist(b.ped) and not IsEntityDead(b.ped) then
                        local bp = GetEntityCoords(b.ped)
                        local d = #(bp - pos)
                        if d < (sep * 0.75) then
                            needSpace = true
                            away = away + (pos - bp)
                        end
                    end
                end
                local dest
                if needSpace then
                    away = away * (1.0 / math.max(0.001, #(away)))
                    dest = pos + away * math.random(4, 8)
                else
                    dest = randomNearbyPoint(pos, 12)
                end
                -- déplacer calmement via navmesh
                TaskFollowNavMeshToCoord(ped, dest.x, dest.y, dest.z, 1.0, math.random(1200, 2400), 0.5, true, 0)
                SetPedKeepTask(ped, true)
                a._nextStep = now + math.random(3800, 7200)
            end

            -- 3) Anti blocage: s'assurer qu'il n'essaie pas de fuir trop loin
            if #(pos - cfg.center) > (cfg.spawnRadius - 2.0) then
                local back = cfg.center + (pos - cfg.center) * 0.6
                TaskFollowNavMeshToCoord(ped, back.x, back.y, back.z, 1.0, 1500, 0.5, true, 0)
            end
        end
        ::continue::
    end
end

local function retargetAndAttack(Z)
    local cfg = Z.cfg
    local now = GetGameTimer()
    if (now - (Z.lastAttackRefresh or 0)) < (Config.AttackRefreshMs or 1500) then return end
    Z.lastAttackRefresh = now

    local targets = playersInZone(cfg)
    if #targets == 0 then return end

    -- Construire une liste de cibles viables (non-coma)
    local viable = {}
    for _, t in ipairs(targets) do
        local ped = t.ped
        if DoesEntityExist(ped) and not IsPedDeadOrDying(ped) and not IsPedFatallyInjured(ped)  then
            local isLocal = (t.pid == PlayerId())
            local coma = isLocal and isPlayerComa() or isOtherPedComa(ped)
            if not coma then
                table.insert(viable, ped)
            end
        end
    end
    if #viable == 0 then return end

    for _, a in ipairs(Z.animals) do
        local ped = a.ped
        if ped and DoesEntityExist(ped) and not IsEntityDead(ped) then
            if not IsPedInCombat(ped, 0) or (GetGameTimer() - (a.lastCombat or 0) > (Config.AttackRefreshMs or 1500)) then
                -- choisir la cible la plus proche
                local myPos = GetEntityCoords(ped)
                local best, bestd = nil, 9e9
                for _, tp in ipairs(viable) do
                    local d = #(GetEntityCoords(tp) - myPos)
                    if d < bestd then bestd, best = d, tp end
                end

                local scale = Config.AggressivenessScale or 1.0
                if scale > 1.0 then scale = 1.0 end
                if scale < 0.0 then scale = 0.0 end
                local go = (Config.Aggressive ~= false) and (math.random() <= scale)
                if best and go then
                    TaskCombatPed(ped, best, 0, 16)
                    SetPedKeepTask(ped, true)
                    a.lastCombat = GetGameTimer()
                else
                    ClearPedTasks(ped)
                end
            end
        end
    end
end

local function zoneClear(Z, keepCadavers)
    for i = #Z.animals, 1, -1 do
        local a = Z.animals[i]
        if a.ped and DoesEntityExist(a.ped) then
            if keepCadavers and IsEntityDead(a.ped) then
                -- garder le cadavre, gestion via cleanup
            else
                deleteAnimal(a.ped)
                table.remove(Z.animals, i)
            end
        else
            table.remove(Z.animals, i)
        end
    end
end

--========================
-- Boucle contrôleur par zone
--========================
CreateThread(function()
    while true do
        local dt = Config.ClientTickMs or 750
        for zoneId, Z in pairs(ClientZones) do
            if Z.controller then
                if Z.state == 'pre' then
                    zoneEnsurePopulation(Z)
                    zoneCull(Z)
                    zoneCleanupCadavers(Z)
                    setAnimalsInvincible(Z, true)  -- invincibles en attente [#5]
                    idleBehavior(Z)
                elseif Z.state == 'active' then
                    zoneEnsurePopulation(Z)
                    zoneCull(Z)
                    zoneCleanupCadavers(Z)
                    setAnimalsInvincible(Z, false) -- vulnérables [#4]
                    retargetAndAttack(Z)
                elseif Z.state == 'blocked' then
                    zoneClear(Z, true)
                elseif Z.state == 'despawn' then
                    zoneClear(Z, true) -- garder cadavres
                else -- idle
                    zoneEnsurePopulation(Z)
                    zoneCull(Z)
                    zoneCleanupCadavers(Z)
                    setAnimalsInvincible(Z, true)
                    idleBehavior(Z)
                end
            end
        end
        Wait(dt)
    end
end)

--===========================================================
-- Événements serveur -> client
--===========================================================
RegisterNetEvent('qb-animalzones:client:setState', function(zoneId, state)
    local Z = ensureZoneInstance(zoneId, Config.ZonesById and Config.ZonesById[zoneId])
    if not Z then return end
    if Z.state == state then return end
    Z.state = state
    if state == 'active' then
        setAnimalsInvincible(Z, false)
    elseif state == 'pre' then
        clearCombatTasks(Z)       -- stop agressions en attente
        setAnimalsInvincible(Z, true)
        idleBehavior(Z)
    elseif state == 'blocked' then
        zoneClear(Z, true)
    elseif state == 'despawn' then
        zoneClear(Z, true) -- garder cadavres
    else -- idle
        clearCombatTasks(Z)       -- stop agressions en sortie de zone
        setAnimalsInvincible(Z, true)
        idleBehavior(Z)
    end
end)

RegisterNetEvent('qb-animalzones:client:takeControl', function(zoneId, cfg)
    local Z = ensureZoneInstance(zoneId, cfg)
    Z.controller = true
    adoptExistingAnimals(Z) -- important à la prise de contrôle
    debug(("take control %s"):format(zoneId))
end)

RegisterNetEvent('qb-animalzones:client:releaseControl', function(zoneId)
    local Z = ClientZones[zoneId]
    if not Z then return end
    debug(("release control %s"):format(zoneId))
    -- Nettoyage: supprimer vivants immédiatement, planifier la suppression des cadavres après le délai restant [#4]
    local now = GetGameTimer()
    local keepMs = (Config.CadaverDespawnMs or 120000)
    for idx = #Z.animals, 1, -1 do
        local a = Z.animals[idx]
        if a.ped and DoesEntityExist(a.ped) then
            if IsEntityDead(a.ped) then
                if not a.deathTs then a.deathTs = now end
                local delay = keepMs - (now - a.deathTs)
                if delay < 0 then delay = 0 end
                local ped = a.ped
                SetTimeout(delay, function()
                    if DoesEntityExist(ped) then
                        deleteAnimal(ped)
                    end
                end)
                table.remove(Z.animals, idx)
            else
                deleteAnimal(a.ped)
                table.remove(Z.animals, idx)
            end
        else
            table.remove(Z.animals, idx)
        end
    end
    Z.controller = false
    Z.state = 'idle'
end)

--===========================================================
-- Heartbeat -> serveur (position/états) [OneSync]
--===========================================================
CreateThread(function()
    while true do
        local ped = PlayerPedId()
        local pos = GetEntityCoords(ped)
        local job = getJobName()
        local coma = isPlayerComa()

        -- calculer les zones proches et internes
        local inside, near = {}, {}
        for _, z in ipairs(Config.Zones) do
            local d = #(pos - z.center)
            if d <= z.radius then
                inside[#inside+1] = z.id
            elseif d <= (z.radius + (Config.PreSpawnBuffer or 120.0)) then
                near[#near+1] = z.id
            end
        end

        TriggerServerEvent('qb-animalzones:server:heartbeat', {
            pos = pos, job = job, coma = coma, inside = inside, near = near, ts = GetGameTimer()
        })
        Wait(Config.HeartbeatMs or 2000)
    end
end)

--===========================================================
-- Initialisation des zones côté client
--===========================================================
CreateThread(function()
    -- Table d'accès rapide aux configs par id
    local byId = {}
    for _,z in ipairs(Config.Zones) do
        byId[z.id] = z
        ensureZoneInstance(z.id, z)
    end
    Config.ZonesById = byId -- cache
end)

--===========================================================
-- Nettoyage si la ressource s'arrête (sécurité)
--===========================================================
AddEventHandler('onResourceStop', function(res)
    if res ~= GetCurrentResourceName() then return end
    for _, Z in pairs(ClientZones) do
        for i = #Z.animals, 1, -1 do
            local a = Z.animals[i]
            if a.ped and DoesEntityExist(a.ped) then
                SetEntityAsMissionEntity(a.ped, true, true)
                DeleteEntity(a.ped)
            end
        end
        Z.animals = {}
        Z.state = 'idle'
        Z.controller = false
    end
end)

CreateThread(function()
    local onesync = GetConvar('onesync', 'off')
    if onesync == 'off' then
        -- We keep working, but we avoid any assumptions about multiple players/network migration
    end
end)
